<?php

namespace Tests\Feature;

use App\Constants\UserConstants;
use App\Models\Image;
use App\Models\Product;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Tests\TestCase;

class UserControllerTest extends TestCase
{
    use RefreshDatabase;

    public function test_authenticated_user_can_upload_a_new_cover_image(): void
    {
        Storage::fake('public');

        $product = Product::factory()->create();
        $product_id = $product->product_id;
        $key = $product->key;
        $user = User::factory()->create();
        $this->actingAs($user, 'api');

        $response = $this->postJson(route('users.set-cover'), [
            'cover' => $cover = UploadedFile::fake()->image('photo.jpg'),
        ], [
            'product' => $product_id,
            'product-key' => $key,
        ]);

        Storage::disk('public')->assertExists("covers/{$cover->hashName()}");
        $response->assertStatus(200);
    }

    public function test_authenticated_user_can_set_existed_cover(): void
    {
        $product = Product::factory()->create();
        $product_id = $product->product_id;
        $key = $product->key;
        $user = User::factory()->create();
        $cover = Image::factory()->create(['user_id' => $user->user_id, 'type' => UserConstants::IMAGE_TYPE_COVER]);
        $this->actingAs($user, 'api');

        $response = $this->postJson(route('users.set-cover'), [
            'cover_id' => $cover->img_id,
        ], [
            'product' => $product_id,
            'product-key' => $key,
        ]);

        $response->assertStatus(200);
        $this->assertDatabaseHas('users', ['cover' => $cover->img_id]);
    }

    public function test_authenticated_user_can_get_all_covers(): void
    {
        $user = User::factory()->create();
        $product = Product::factory()->create();
        $product_id = $product->product_id;
        $key = $product->key;
        Image::factory()->create(['user_id' => $user->user_id, 'type' => UserConstants::IMAGE_TYPE_COVER]);
        Image::factory()->create(['user_id' => $user->user_id, 'type' => UserConstants::IMAGE_TYPE_COVER]);
        $this->actingAs($user, 'api');

        $response = $this->getJson(route('users.covers'), [
            'product' => $product_id,
            'product-key' => $key,
        ]);

        $response->assertStatus(200);
        $response->assertJsonCount(2, 'covers');
    }

    public function test_authenticated_user_can_upload_a_new_photo_image(): void
    {
        Storage::fake('public');

        $product = Product::factory()->create();
        $product_id = $product->product_id;
        $key = $product->key;
        $user = User::factory()->create();
        $this->actingAs($user, 'api');

        $response = $this->postJson(route('users.set-main-photo'), [
            'photo' => $cover = UploadedFile::fake()->image('photo.jpg'),
        ], [
            'product' => $product_id,
            'product-key' => $key,
        ]);

        Storage::disk('public')->assertExists("photos/{$cover->hashName()}");
        $response->assertStatus(200);
    }

    public function test_authenticated_user_can_set_existed_photo(): void
    {
        $product = Product::factory()->create();
        $product_id = $product->product_id;
        $key = $product->key;
        $user = User::factory()->create();
        $photo = Image::factory()->create(['user_id' => $user->user_id, 'type' => UserConstants::IMAGE_TYPE_PROFILE]);
        $this->actingAs($user, 'api');

        $response = $this->postJson(route('users.set-main-photo'), [
            'photo_id' => $photo->img_id,
        ], [
            'product' => $product_id,
            'product-key' => $key,
        ]);

        $response->assertStatus(200);
        $this->assertDatabaseHas('users', ['image' => $photo->img_id]);
    }

    public function test_authenticated_user_can_get_all_photos(): void
    {
        $user = User::factory()->create();
        $product = Product::factory()->create();
        $product_id = $product->product_id;
        $key = $product->key;
        Image::factory()->create(['user_id' => $user->user_id, 'type' => UserConstants::IMAGE_TYPE_PROFILE]);
        Image::factory()->create(['user_id' => $user->user_id, 'type' => UserConstants::IMAGE_TYPE_PROFILE]);
        $this->actingAs($user, 'api');

        $response = $this->getJson(route('users.photos'), [
            'product' => $product_id,
            'product-key' => $key,
        ]);

        $response->assertStatus(200);
        $response->assertJsonCount(2, 'photos');
    }
}
