<?php

namespace Tests\Feature;

use Tests\TestCase;
use App\Models\Product;
use App\Enums\SocialPlatforms;
use Illuminate\Foundation\Testing\RefreshDatabase;

class PlatformControllerTest extends TestCase
{
    use RefreshDatabase;

    public function test_list_platforms(): void
    {        
        $product = Product::factory()->create();
        $product_id = $product->product_id;
        $key = $product->key;

        $response = $this->getJson(route('auth.social-platforms'), [
            'product' => $product_id,
            'product-key' => $key,
        ]);

        $response->assertStatus(200);
        $response->assertJson([
            'success' => true,
            'platforms' => SocialPlatforms::supported(),
        ]);
    }

    public function test_redirect_to_provider(): void
    {
        $provider = SocialPlatforms::GITHUB;

        $product = Product::factory()->create();
        $product_id = $product->product_id;
        $key = $product->key;

        $response = $this->getJson(route('auth.login.socialite', [
            'provider' => $provider,
        ]),
            [
                'product' => $product_id,
                'product-key' => $key,
            ]
        );

        $response->assertStatus(302);
        $response->assertRedirect();
    }

    public function test_redirect_to_provider_with_invalid_product(): void
    {
        $provider = 'github';
        $product_id = 'invalid_product_id';
        $key = 'invalid_key';

        $response = $this->getJson(route('auth.login.socialite', [
            'provider' => $provider,
        ]),
            ['product' => $product_id, 'product-key' => $key]
        );

        $response->assertStatus(400);
        $response->assertJson([
            'success' => false,
            'message' => __('Invalid product'),
        ]);
    }

    public function test_redirect_to_provider_with_unsupported_provider(): void
    {
        $provider = 'unsupported_provider';
        
        $product = Product::factory()->create();
        $product_id = $product->product_id;
        $key = $product->key;

        $response = $this->getJson(route('auth.login.socialite', [
            'provider' => $provider,
        ]),
            [
                'product' => $product_id,
                'product-key' => $key,
            ]
        );

        $response->assertStatus(400);
        $response->assertJson([
            'success' => false,
            'message' => __(':provider is not supported', ['provider' => $provider]),
        ]);
    }
}
