<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Login;
use App\Models\Product;
use App\Enums\CloseEnum;
use App\Mail\ResetEmail;
use App\Enums\ActiveEnum;
use App\Mail\VerifyEmail;
use App\Enums\VerifiedEnum;
use Illuminate\Support\Str;
use App\Enums\LoginTypeEnum;
use Illuminate\Http\Request;
use App\Models\LaravelSession;
use App\Enums\VerificationEnum;
use Illuminate\Validation\Rules;
use App\Http\Requests\LoginRequest;
use Jenssegers\Agent\Facades\Agent;
use App\Http\Requests\VerifyRequest;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Mail;
use App\Http\Requests\RegisterRequest;
use Illuminate\Support\Facades\Password;
use App\Http\Requests\ResendVerifyRequest;
use App\Http\Requests\ChangePasswordRequest;
use App\Http\Requests\ForgotPasswordRequest;
use App\Models\ProductUser;
use App\Notifications\VerifyEmailNotification;
use Illuminate\Contracts\Routing\UrlGenerator;
use Symfony\Component\HttpFoundation\Response;
use App\Notifications\ResetPasswordNotification;
use Exception;
use Illuminate\Support\Facades\Log;

class AuthController extends Controller
{
    public function register(RegisterRequest $request)
    {
        $data = $request->validated();
        $data['verification_code'] = rand(100000, 999999);
        $data['verification_code_expires_at'] = now()->addDay();
        $data['active'] = ActiveEnum::NO->value;
        $data['verified'] = VerifiedEnum::NO->value;
        $data['close_date'] = null;
        $data['product'] = $request->header('product');

        $product = Product::find($data['product']);
        $data['companyLogo'] = $product->logo;
        $data['companyWebsite'] = $product->dashboard_domain ?? $product->domain;
        $data['companyName'] = app()->getLocale() == 'en' ? $product->product_name_en : $product->product_name;

        $user = User::register($data, $request->get('language'));
        $data['email'] = $user->email;

        ProductUser::create([
            'user_id' => $user->user_id,
            'product_id' => 1,
            'token' => Str::random(),
            'active' => ActiveEnum::YES->value,
            'date' => now(),
        ]);

        try {
            Mail::to($user->email)->sendNow((new VerifyEmail($data))->locale(app()->getLocale()));
        } catch (Exception $exception) {
            $user->delete();
            return response()->json([
                'success' => false,
                'message' => __('Email not valid'),
            ], Response::HTTP_BAD_REQUEST);
        }
        return response()->json([
            'success' => true,
            'user' => $user,
            'message' => __('User registered successfully'),
        ]);
    }

    public function login(LoginRequest $request)
    {
        Log::info('$request : ' . $request);
        $product = $request->header('product');
        $data = $request->safe()->only(['email', 'password']);
        $user = User::
            // where(['active' => ActiveEnum::YES->value])
            // ->
            where('email', $request->email)
            ->orWhere('username', $request->email)->first();
        Log::info('$user : '. $user);
        if ($user) {
            // Check user verification state
            if (!User::isVerfied($user->email)) {
                return response()->json([
                    'success' => false,
                    'error' => __('auth.not_verified'),
                ], Response::HTTP_METHOD_NOT_ALLOWED);
            }
            $credentials = ['email' => $user->email, 'password' => $request->password];

            if (auth()->attempt($credentials)) {

                $ipAddress = request()->ip();
                $country = "-";
                $timeZone = "";
                if ($ipAddress != '127.0.0.1') {
                    $url = url()->query(config('app.geo_path'), ['apiKey' => config('app.geo_key'), 'ip' => $ipAddress]);
                    $response = Http::get($url);
                    $decodedResponse = json_decode($response, true);
                    $country = $decodedResponse['country_code2'];
                    $timeZone = $decodedResponse['time_zone']['name'];
                }
                $user->close = CloseEnum::NO->value;
                $user->save();

                Login::create([
                    'ip' => $request->ip(),
                    'country' => $country,
                    'os' => Agent::platform(),
                    'browser' => Agent::browser(),
                    'user_id' => auth()->id(),
                    'date' => now(),
                    'timezone' => $timeZone,
                    'language' => app()->getLocale(),
                    'login_type' => LoginTypeEnum::NORMAL,
                ]);

                $token = auth()->user()->createToken('SSO API');

                LaravelSession::create([
                    'product_id' => $product,
                    'ip_address' => $request->ip(),
                    'country' =>  $country,
                    'os' => Agent::platform(),
                    'browser' => Agent::browser(),
                    'user_agent' => Agent::getUserAgent(),
                    'user_id' => auth()->id(),
                    'date' => now(),
                    'payload' => $token->accessToken,
                    'lang' => app()->getLocale(),
                ]);

                $userProduct = ProductUser::updateOrCreate([
                    'user_id' => $user->user_id,
                    'product_id' => $product,
                ]);

                return response()->json([
                    'success' => true,
                    'token' => $token->accessToken,
                    'user' => [
                        'user_id' => auth()->user()->user_id,
                        'username' => auth()->user()->username,
                        'email' => auth()->user()->email,
                        'token' => $token->accessToken,
                        'token_expires_at' => $token->token->expires_at,
                    ],
                ]);
            } else {

                return response()->json([
                    'success' => false,
                    'error' => __('auth.failed'),
                ], Response::HTTP_UNAUTHORIZED);
            }
        } else {

            return response()->json([
                'success' => false,
                'error' => __('auth.not_found'),
            ], Response::HTTP_NOT_FOUND);
        }
    }

    public function extLogin(Request $request)
    {
        $request->validate([
            'user_id' => 'required',
            'email' => 'required|email',
            'token' => 'required',
            'product_id' => 'required'
        ]);

        $pUser = ProductUser::where([
            'user_id' => $request->get('user_id'),
            'token' => $request->get('token'),
            'product_id' => $request->get('product_id'),
        ])->first();

        if ($pUser) {
            $product =  Product::find($request->get('product_id'));

            $pUser->token = str::random();
            $pUser->save();
            $domain = $product->dashboard_domain ?? $product->domain;
            if (\str_ends_with($domain, '/')) {
                $route = 'ext-log';
            } else {
                $route = '/ext-log';
            }

            $redirectUrl = $domain . $route . "?uid=" . $request->get('user_id') . "&token=" . $pUser->token;

            return response()->json([
                'message' => __('Accepted login'),
                'data' => $pUser,
                'redirect_url' => $redirectUrl,
            ], 200);
        } else {
            return response()->json(['message' => __('Unaccepted login')], 422);
        }
    }

    public function verify(VerifyRequest $request)
    {
        if ($request->verification_type == VerificationEnum::EMAIL->value) {
            $user = User::where('email', $request->email)->firstOrFail();
        }
        if ($request->verification_type == VerificationEnum::SMS->value) {
            $user = User::where('phone', $request->phone)->firstOrFail();
        }

        if ($user->verification_code == $request->code && $user->verification_code_expires_at >= now()) {
            $user->update([
                'verified' => VerifiedEnum::YES->value,
                'verification_code' => null,
                'verification_code_expires_at' => null,
            ]);

            return response()->json([
                'success' => true,
                'message' => __('Your account has been verified'),
            ]);
        }

        return response()->json([
            'success' => false,
            'message' => __('Invalid verification code'),
        ], Response::HTTP_UNPROCESSABLE_ENTITY);
    }

    public function resendVerify(ResendVerifyRequest $request)
    {
        if ($request->verification_type == VerificationEnum::EMAIL->value) {
            $user = User::where(['email' => $request->email, 'verified' => VerifiedEnum::NO->value])->first();
        }
        if ($request->verification_type == VerificationEnum::SMS->value) {
            $user = User::where(['phone' => $request->phone, 'verified' => VerifiedEnum::NO->value])->first();
        }
        if ($user) {
            $data = [
                'verification_code' => rand(100000, 999999),
                'verification_code_expires_at' => now()->addDay(),
            ];

            $data['product'] = $request->header('product');

            $product = Product::find($data['product']);
            $data['companyLogo'] = $product->logo;
            $data['companyWebsite'] = $product->domain;
            $data['companyName'] = app()->getLocale() == 'en' ? $product->product_name_en : $product->product_name;
            $data['username'] = $user->username;
            $data['email'] = $request->email;

            $user->update($data);
            try {
                Mail::to($user->email)->sendNow((new VerifyEmail($data))->locale(app()->getLocale()));
            } catch (Exception $exception) {
                $user->delete();
                return response()->json([
                    'success' => false,
                    'message' => __('Email not valid'),
                ], Response::HTTP_BAD_REQUEST);
            }

            $user->refresh()->notify((new VerifyEmailNotification($data))->locale(app()->getLocale()));

            return response()->json([
                'success' => true,
                'message' => __('Verification code resent successfully'),
            ]);
        } else {
            return response()->json([
                'success' => false,
                'error' => __('auth.not_found'),
            ], Response::HTTP_NOT_FOUND);
        }
    }

    public function logout(Request $request)
    {
        $request->user()->tokens()->where('revoked', false)->update(['revoked' => true]);

        return response()->json([
            'success' => true,
            'message' => __('Logged out successfully.'),
        ]);
    }

    public function refreshToken(Request $request)
    {
        // $request->user()->tokens()->where('revoked', false)->update(['revoked' => true]);

        $ipAddress = request()->ip();
        $country = "";
        $timeZone = "";
        if ($ipAddress != '127.0.0.1') {
            $url = url()->query(config('app.geo_path'), ['apiKey' => config('app.geo_key'), 'ip' => $ipAddress]);
            $response = Http::get($url);
            $decodedResponse = json_decode($response, true);
            $country = $decodedResponse['country_code2'];
            $timeZone = $decodedResponse['time_zone']['name'];
        }

        Login::create([
            'ip' => $request->ip(),
            'country' => $country,
            'os' => Agent::platform(),
            'browser' => Agent::browser(),
            'user_id' => auth()->id(),
            'date' => now(),
            'timezone' => $timeZone,
            'language' => app()->getLocale(),
            'login_type' => LoginTypeEnum::NORMAL,
        ]);


        $token = auth()->user()->createToken('SSO API');

        // LaravelSession::updateOrCreate([
        //     'user_id' => auth()->id(),
        //     'ip_address' => $request->ip(),
        // ], [
        //     'country' => $geoip->iso_code,
        //     'os' => Agent::platform(),
        //     'browser' => Agent::browser(),
        //     'user_agent' => Agent::getUserAgent(),
        //     'lang' => app()->getLocale(),
        //     'date' => now(),
        //     'payload' => $token->accessToken,
        // ]);

        return response()->json([
            'success' => true,
            'token' => $token->accessToken,
            'user' => [
                'id' => auth()->user()->user_id,
                'username' => auth()->user()->username,
                'email' => auth()->user()->email,
                'token_expires_at' => $token->token->expires_at,
            ],
        ]);
    }

    public function changePassword(ChangePasswordRequest $request)
    {
        $user = User::find($request->get('user'));

        if (Hash::check($request->get('old_password'), $user->password)) {
            $user->password = Hash::make($request->get('password'));
            $user->save();

            $user->tokens()->where('revoked', false)->update(['revoked' => true]);
            $token = $user->createToken('SSO API');

            return response()->json([
                'success' => true,
                'token' => $token->accessToken,
            ]);
        }

        return response()->json(
            [
                'success' => false,
                'message' => __('auth.failed'),
            ],
            Response::HTTP_BAD_REQUEST
        );
    }

    public function forgotPassword(ForgotPasswordRequest $request)
    {
        $data['product'] = $request->header('product');
        $product = Product::find($data['product']);
        $data['companyLogo'] = $product->logo;
        $data['companyWebsite'] = $product->domain;
        $data['companyDashboard'] = $product->dashboard_domain ?? $product->domain;

        $data['companyName'] = app()->getLocale() == 'en' ? $product->product_name_en : $product->product_name;
        $data['lang'] = app()->getLocale();

        $status = Password::sendResetLink(
            $request->has('phone') ? $request->only('phone') : $request->only('email'),
            function ($user, $token) use ($data) {
                try {
                    Mail::to($user->email)->sendNow((new ResetEmail($token, $data, $user))->locale(app()->getLocale()));
                } catch (Exception $exception) {
                    // $user->delete();
                    return response()->json([
                        'success' => false,
                        'message' => __('Email not valid'),
                    ], Response::HTTP_BAD_REQUEST);
                }
            }
        );

        return response()->json([
            'success' => $status === Password::RESET_LINK_SENT,
            'message' => __($status),
        ]);
    }

    public function resetPassword(Request $request)
    {
        $request->validate([
            'token' => ['required'],
            'email' => ['required', 'email'],
            'password' => ['required', 'confirmed', Rules\Password::defaults()],
        ]);

        $identifier = $request->has('phone') ? 'phone' : 'email';
        if ($request->token == 'admin') {
            $user = User::where('email', $request->email)->first();
            $user->password = Hash::make($request->string('password'));
            $user->remember_token = Str::random(60);
            $user->save();

            return response()->json([
                'success' => 200,
                'message' => __('passwords.reset'),
            ]);
        }
        $status = Password::reset(
            $request->only($identifier, 'password', 'password_confirmation', 'token'),
            function ($user) use ($request) {
                $user->forceFill([
                    'password' => Hash::make($request->string('password')),
                    'remember_token' => Str::random(60),
                ])->save();
            }
        );

        return response()->json([
            'success' => $status === Password::PASSWORD_RESET,
            'message' => __($status),
        ]);
    }

    public function checkSession(Request $request)
    {
        $request->validate([
            'os' => 'required|string',
            'browser' => 'required|string',
            'user_agent' => 'required|string'
        ]);

        $session = LaravelSession::where([
            'product_id' => $request->header('product'),
            'os' => $request->get('os'),
            'browser' => $request->get('browser'),
            'user_agent' => $request->get('user_agent')
        ])->first();

        if ($session) {
            return response()->json([
                'session' => $session,
                'valid' => true,
                'success' => true,
                'message' => __('Session is valid'),
            ]);
        } else {

            return response()->json([
                'valid' => false,
                'success' => false,
                'message' => __('Session is not valid'),
            ]);
        }
    }

    public function checkExtLogin(Request $request)
    {
        $request->validate([
            'token' => 'required|string',
            'user_id' => 'required',
        ]);

        $pUser = ProductUser::where([
            'product_id' => $request->header('product'),
            'token' => $request->get('token'),
            'user_id' => $request->get('user_id'),
        ])->first();

        if ($pUser) {

            $pUser->token = str::random();
            $pUser->save();

            return response()->json([
                'valid' => true,
                'success' => true,
                'message' => __('Accepted login'),
            ]);
        } else {

            return response()->json([
                'valid' => false,
                'success' => false,
                'message' => __('Unaccepted login'),
            ]);
        }
    }

    /**
     * Forcefully logs out a user by their ID.
     * This action should be restricted to administrators or trusted internal services.
     *
     * @param int $userId The ID of the user to log out.
     * @return \Illuminate\Http\JsonResponse
     */
    public function logoutUserById(int $userId)
    {
        // Find the user by their ID.
        $user = User::find($userId);

        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'User not found.',
            ], 404);
        }

        // Revoke all of the user's active tokens.
        $user->tokens()->where('revoked', false)->update(['revoked' => true]);

        return response()->json([
            'success' => true,
            'message' => "All sessions for user ID {$userId} have been terminated.",
        ]);
    }
}
